<?php
require_once 'config.php';

// Cek apakah PhpSpreadsheet tersedia (untuk .xlsx/.xls/.csv)
$hasPhpSpreadsheet = false;
if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    require_once __DIR__ . '/vendor/autoload.php';
    if (class_exists('\PhpOffice\PhpSpreadsheet\IOFactory')) $hasPhpSpreadsheet = true;
}

/**
 * Helper: kirim response JSON dan exit
 */
function json($data){
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data);
    exit;
}

/**
 * Ambil ID tim berdasarkan nama. Jika belum ada, buat baru.
 */
function getTeamId($conn, $name){
    $name = trim($name);
    $q = mysqli_prepare($conn, "SELECT id FROM teams WHERE name=?");
    mysqli_stmt_bind_param($q, "s", $name);
    mysqli_stmt_execute($q);
    $r = mysqli_stmt_get_result($q);
    if ($d = mysqli_fetch_assoc($r)) {
        mysqli_stmt_close($q);
        return $d['id'];
    }
    mysqli_stmt_close($q);

    $q = mysqli_prepare($conn, "INSERT INTO teams(name) VALUES(?)");
    mysqli_stmt_bind_param($q, "s", $name);
    mysqli_stmt_execute($q);
    mysqli_stmt_close($q);
    return mysqli_insert_id($conn);
}

/**
 * Ambil ID pegawai berdasarkan nama. Jika belum ada, buat baru.
 * NOTE: pegawai sekarang **tidak terikat tim**, sehingga kita hanya menyimpan nama.
 * Parameter $teamId tetap ada untuk kompatibilitas pemanggilan, tapi tidak digunakan.
 */
function getEmployeeId($conn, $name, $teamId = null){
    $name = trim($name);

    $q = mysqli_prepare($conn, "SELECT id FROM employees WHERE name=?");
    mysqli_stmt_bind_param($q, "s", $name);
    mysqli_stmt_execute($q);
    $r = mysqli_stmt_get_result($q);
    if ($d = mysqli_fetch_assoc($r)) {
        mysqli_stmt_close($q);
        return $d['id'];
    }
    mysqli_stmt_close($q);

    // Insert only name (pegawai global)
    $q = mysqli_prepare($conn, "INSERT INTO employees(name) VALUES(?)");
    mysqli_stmt_bind_param($q, "s", $name);
    mysqli_stmt_execute($q);
    mysqli_stmt_close($q);
    return mysqli_insert_id($conn);
}

/**
 * Normalisasi string header untuk pencocokan
 */
function normalizeHeader($h){
    $h = (string)$h;
    // Hapus BOM
    $h = preg_replace("/^\xEF\xBB\xBF/", '', $h);
    $h = trim(mb_strtolower($h, 'UTF-8'));
    if(function_exists('iconv')){
        $tmp = @iconv('UTF-8', 'ASCII//TRANSLIT', $h);
        if($tmp !== false) $h = $tmp;
    }
    $h = preg_replace('/[^a-z0-9]+/i', ' ', $h);
    $h = preg_replace('/\s+/', ' ', $h);
    return trim($h);
}

/**
 * Mapping header
 */
function mapHeaderToCanonical($rawHeader){
    $h = normalizeHeader($rawHeader);
    if($h === '') return null;

    // cek token utama
    if (strpos($h,'tim') !== false || strpos($h,'team') !== false) return 'nama_tim';
    if (strpos($h,'pegawai') !== false || strpos($h,'employee') !== false || $h === 'nama') return 'nama_pegawai';
    if (strpos($h,'kegiatan') !== false || strpos($h,'activity') !== false) {
        // pastikan bukan sub
        if (strpos($h,'sub') !== false) return 'nama_sub_kegiatan';
        return 'nama_kegiatan';
    }
    if (strpos($h,'sub') !== false || strpos($h,'sub kegiatan') !== false || strpos($h,'sub_activity') !== false) return 'nama_sub_kegiatan';
    if (strpos($h,'target') !== false || strpos($h,'jumlah') !== false || strpos($h,'volume') !== false || strpos($h,'qty') !== false) return 'target';
    if (strpos($h,'satuan') !== false || strpos($h,'unit') !== false) return 'satuan';
    if (strpos($h,'bulan') !== false || strpos($h,'month') !== false) return 'bulan';
    if (strpos($h,'tahun') !== false || strpos($h,'year') !== false) return 'tahun';
    if (strpos($h,'deadline') !== false || strpos($h,'tanggal') !== false || strpos($h,'tgl') !== false || strpos($h,'due') !== false || strpos($h,'date') !== false) return 'deadline';

    return null;
}

/**
 * Detect delimiter sederhana
 */
function detectCsvDelimiter($line){
    $candidates = [',',';',"\t"];
    $best = ',';
    $bestCount = 0;
    foreach($candidates as $d){
        $cnt = substr_count($line, $d);
        if($cnt > $bestCount){ $best = $d; $bestCount = $cnt; }
    }
    return $best;
}

/**
 * Parse deadline fleksibel:
 * - Menerima serial Excel (jika PhpSpreadsheet tersedia)
 * - Menerima format lengkap (Y-m-d, d-m-Y, slashes)
 * - Menerima MM-YYYY atau YYYY-MM atau "Maret 2026" atau "March 2026"
 * - Jika hanya bulan/tahun -> mengembalikan YYYY-MM-01 (agar valid untuk kolom DATE)
 * - Jika tidak bisa di-parse -> mengembalikan string asli
 */
function parseDeadline($raw, $hasPhpSpreadsheet = false){
    $raw = trim((string)$raw);
    if($raw === '') return null;

    // 1) Excel serial number (terutama saat PhpSpreadsheet membaca cell tanpa format)
    if(is_numeric($raw) && $hasPhpSpreadsheet){
        try {
            $dt = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject((float)$raw);
            return $dt->format('Y-m-d');
        } catch(Exception $ex){}
    }

    // normalisasi
    $r = trim($raw);
    // replace some separators
    $r2 = str_replace(['/', '\\', '.'], '-', $r);

    // 2) Full date formats that strtotime bisa tangani langsung
    $try = strtotime($r2);
    if($try !== false){
        // dapatkan Y-m-d
        return date('Y-m-d', $try);
    }

    // 3) Cek pola MM-YYYY, M-YYYY, YYYY-MM
    if(preg_match('/^(\d{1,2})-(\d{4})$/', $r2, $m)){
        $month = intval($m[1]);
        $year  = intval($m[2]);
        if($month >=1 && $month <=12){
            return sprintf('%04d-%02d-01', $year, $month);
        }
    }
    if(preg_match('/^(\d{4})-(\d{1,2})$/', $r2, $m)){
        $year = intval($m[1]);
        $month = intval($m[2]);
        if($month >=1 && $month <=12){
            return sprintf('%04d-%02d-01', $year, $month);
        }
    }

    // 4) Cek nama bulan (Indonesia & English) + optional year
    $mapBulan = [
        'januari'=>1,'februari'=>2,'maret'=>3,'april'=>4,'mei'=>5,'juni'=>6,
        'juli'=>7,'agustus'=>8,'september'=>9,'oktober'=>10,'november'=>11,'desember'=>12,
        'january'=>1,'february'=>2,'march'=>3,'april'=>4,'may'=>5,'june'=>6,
        'july'=>7,'august'=>8,'september'=>9,'october'=>10,'november'=>11,'december'=>12,
        // singkatan
        'jan'=>1,'feb'=>2,'mar'=>3,'apr'=>4,'may'=>5,'jun'=>6,'jul'=>7,'aug'=>8,'sep'=>9,'oct'=>10,'nov'=>11,'dec'=>12
    ];
    $low = mb_strtolower($r, 'UTF-8');
    foreach($mapBulan as $name=>$num){
        if(strpos($low, $name) !== false){
            // coba temukan tahun di teks
            if(preg_match('/\b(19|20)\d{2}\b/', $low, $y)){
                $year = intval($y[0]);
            } else {
                $year = intval(date('Y')); // default ke tahun sekarang jika tidak ada
            }
            return sprintf('%04d-%02d-01', $year, $num);
        }
    }

    // 5) fallback: kembalikan apa adanya (seperti sebelumnya)
    return $raw;
}

/* ================= TEMPLATE DOWNLOAD ROUTE ================= */
if (isset($_GET['action']) && $_GET['action'] === 'download_template') {
    $templatePath = __DIR__ . '/Tamplate Tambah Kegiatan.xlsx';
    if (!file_exists($templatePath)) {
        if (file_exists(__DIR__ . '/Tamplate Tambah Kegiatan.xls')) $templatePath = __DIR__ . '/Tamplate Tambah Kegiatan.xls';
        elseif (file_exists(__DIR__ . '/Tamplate Tambah Kegiatan.csv')) $templatePath = __DIR__ . '/Tamplate Tambah Kegiatan.csv';
    }
    if (!file_exists($templatePath)) {
        header($_SERVER['SERVER_PROTOCOL'] . ' 404 Not Found');
        echo "Template file tidak ditemukan.";
        exit;
    }
    $basename = basename($templatePath);
    $ext = strtolower(pathinfo($basename, PATHINFO_EXTENSION));
    $ct = 'application/octet-stream';
    if ($ext === 'xlsx') $ct = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
    if ($ext === 'xls')  $ct = 'application/vnd.ms-excel';
    if ($ext === 'csv')  $ct = 'text/csv';

    header('Content-Description: File Transfer');
    header('Content-Type: ' . $ct);
    header('Content-Disposition: attachment; filename="' . $basename . '"');
    header('Content-Transfer-Encoding: binary');
    header('Expires: 0');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    header('Content-Length: ' . filesize($templatePath));
    ob_clean();
    flush();
    readfile($templatePath);
    exit;
}

/* ================= API tambahan untuk AJAX (employees & sub activities & activities) ================= */
if(isset($_GET['action'])){

    // GET ALL EMPLOYEES (tidak terikat tim)
    if($_GET['action'] === 'get_employees'){
        $q = mysqli_prepare($conn, "SELECT id, name FROM employees ORDER BY name");
        mysqli_stmt_execute($q);
        $res = mysqli_stmt_get_result($q);
        $out = [];
        while($row = mysqli_fetch_assoc($res)) $out[] = $row;
        json($out);
    }

    /* Robust get_subs:
       - Jika tabel master_sub_kegiatan MENGANDUNG kolom kegiatan_id, kita akan:
         - Bila param 'kegiatan' dikirim -> kembalikan sub terkait kegiatan tersebut (JOIN).
         - Bila param 'kegiatan' tidak dikirim -> kembalikan semua sub.
       - Jika kolom kegiatan_id TIDAK ADA (sub global), kembalikan semua sub langsung.
    */
    if($_GET['action'] === 'get_subs'){
        $keg = isset($_GET['kegiatan']) ? trim($_GET['kegiatan']) : null;

        // cek apakah kolom kegiatan_id ada
        $colCheck = mysqli_query($conn, "SHOW COLUMNS FROM `master_sub_kegiatan` LIKE 'kegiatan_id'");
        $out = [];
        if($colCheck && mysqli_num_rows($colCheck) > 0){
            // kolom ada -> bisa filter by kegiatan jika diberikan
            if($keg){
                $q = mysqli_prepare($conn, "SELECT s.id, s.nama_sub_kegiatan FROM master_sub_kegiatan s JOIN master_kegiatan k ON k.id=s.kegiatan_id WHERE k.nama_kegiatan=? ORDER BY s.nama_sub_kegiatan");
                mysqli_stmt_bind_param($q, "s", $keg);
                mysqli_stmt_execute($q);
                $res = mysqli_stmt_get_result($q);
            } else {
                $res = mysqli_query($conn, "SELECT id, nama_sub_kegiatan FROM master_sub_kegiatan ORDER BY nama_sub_kegiatan");
            }
        } else {
            // kolom tidak ada -> sub bersifat global
            $res = mysqli_query($conn, "SELECT id, nama_sub_kegiatan FROM master_sub_kegiatan ORDER BY nama_sub_kegiatan");
        }

        if($res){
            while($row = mysqli_fetch_assoc($res)) $out[] = $row;
        }
        json($out);
    }

    // ambil activities berdasarkan team name
    if($_GET['action'] === 'get_activities' && isset($_GET['team'])){
        $teamName = trim($_GET['team']);
        $q = mysqli_prepare($conn, "
            SELECT mk.id, mk.nama_kegiatan
            FROM master_kegiatan mk
            JOIN teams t ON t.id = mk.team_id
            WHERE t.name = ?
            ORDER BY mk.nama_kegiatan
        ");
        mysqli_stmt_bind_param($q, "s", $teamName);
        mysqli_stmt_execute($q);
        $res = mysqli_stmt_get_result($q);
        $out = [];
        while($row = mysqli_fetch_assoc($res)) $out[] = $row;
        json($out);
    }
}

/* ================= API utama (save/update/list/detail/delete) ================= */
if(isset($_GET['action'])){

    if($_GET['action']=='save'){
        // =========================================
        // Jika file Excel/CSV diunggah -> PROSES FILE SAJA
        // =========================================
        if(isset($_FILES['excel_file']) && isset($_FILES['excel_file']['tmp_name'])
            && is_uploaded_file($_FILES['excel_file']['tmp_name']) && $_FILES['excel_file']['error'] === UPLOAD_ERR_OK){

            $uploaded = $_FILES['excel_file'];
            $tmpPath = $uploaded['tmp_name'];
            $origName = $uploaded['name'];
            $ext = strtolower(pathinfo($origName, PATHINFO_EXTENSION));

            $rows = []; // akan berisi baris-baris associative (menggunakan header file)

            try {
                // --- Baca file dengan PhpSpreadsheet bila tersedia (.xlsx/.xls/.csv) ---
                if($hasPhpSpreadsheet && in_array($ext, ['xlsx','xls','csv'])) {
                    $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReaderForFile($tmpPath);
                    if($ext === 'csv'){
                        // deteksi delimiter dari isi
                        $firstLine = '';
                        $fh = fopen($tmpPath, 'r');
                        if($fh){
                            $firstLine = fgets($fh);
                            fclose($fh);
                        }
                        $det = detectCsvDelimiter($firstLine ?: ',');
                        if(method_exists($reader, 'setDelimiter')) $reader->setDelimiter($det);
                        if(method_exists($reader, 'setEnclosure')) $reader->setEnclosure('"');
                    }
                    $spreadsheet = $reader->load($tmpPath);
                    $sheet = $spreadsheet->getActiveSheet();
                    $highestRow = $sheet->getHighestDataRow();
                    $highestColumn = $sheet->getHighestDataColumn();
                    $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);

                    // Baca header (baris 1)
                    $rawHeader = [];
                    for($c=1;$c<=$highestColumnIndex;$c++){
                        $val = $sheet->getCellByColumnAndRow($c, 1)->getFormattedValue();
                        $rawHeader[] = trim((string)$val);
                    }

                    // Map header menjadi kunci bahasa Indonesia snake_case (jika tidak dikenali -> null)
                    $mappedHeader = [];
                    foreach($rawHeader as $h){
                        $mappedHeader[] = mapHeaderToCanonical($h);
                    }

                    // Baca data mulai baris 2
                    for($r=2;$r<=$highestRow;$r++){
                        $rowAssoc = [];
                        $empty = true;
                        for($c=1;$c<=$highestColumnIndex;$c++){
                            $key = $mappedHeader[$c-1] ?? null;
                            $cell = $sheet->getCellByColumnAndRow($c, $r);
                            $val = $cell->getValue();

                            // Jika Excel menyimpan tanggal sebagai serial number, konversi ke format Y-m-d
                            if(\PhpOffice\PhpSpreadsheet\Shared\Date::isDateTime($cell) && is_numeric($val)){
                                try {
                                    $dt = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($val);
                                    $val = $dt->format('Y-m-d');
                                } catch(Exception $ex){}
                            }

                            if ($val !== null && $val !== '') $empty = false;

                            // Jika header tidak dikenali (null), simpan sebagai colN agar tidak hilang
                            if(!$key) $key = 'col'.$c;
                            $rowAssoc[$key] = is_string($val) ? trim($val) : $val;
                        }
                        if(!$empty) $rows[] = $rowAssoc;
                    }

                } else {
                    // --- Fallback: baca CSV saja ---
                    if(in_array($ext, ['csv','txt']) || $ext === ''){
                        if (($handle = fopen($tmpPath, "r")) !== FALSE) {
                            // baca first line untuk deteksi delimiter
                            $firstLine = fgets($handle);
                            if($firstLine === false) throw new Exception("CSV kosong atau header tidak terbaca.");
                            $delimiter = detectCsvDelimiter($firstLine);
                            rewind($handle);
                            $rawHeader = fgetcsv($handle, 0, $delimiter);
                            if($rawHeader === FALSE) throw new Exception("CSV kosong atau header tidak terbaca.");
                            $rawHeader = array_map(function($h){ return trim((string)$h); }, $rawHeader);
                            // Map header
                            $mappedHeader = [];
                            foreach($rawHeader as $h) $mappedHeader[] = mapHeaderToCanonical($h);

                            while (($data = fgetcsv($handle, 0, $delimiter)) !== FALSE) {
                                $rowAssoc = []; $empty = true;
                                for($i=0;$i<count($mappedHeader);$i++){
                                    $key = $mappedHeader[$i] ?? null;
                                    if(!$key) $key = 'col'.($i+1);
                                    $val = isset($data[$i]) ? $data[$i] : null;
                                    if ($val !== null && $val !== '') $empty = false;
                                    $rowAssoc[$key] = is_string($val) ? trim($val) : $val;
                                }
                                // jika baris punya kolom lebih dari header, simpan tambahan sebagai colN
                                if(count($data) > count($mappedHeader)){
                                    for($j = count($mappedHeader); $j < count($data); $j++){
                                        $rowAssoc['col'.($j+1)] = is_string($data[$j]) ? trim($data[$j]) : $data[$j];
                                        if ($data[$j] !== null && $data[$j] !== '') $empty = false;
                                    }
                                }
                                if(!$empty) $rows[] = $rowAssoc;
                            }
                            fclose($handle);
                        } else {
                            throw new Exception("Gagal membuka file CSV.");
                        }
                    } else {
                        throw new Exception("PhpSpreadsheet tidak tersedia. Untuk mengunggah .xlsx/.xls, pasang phpoffice/phpspreadsheet atau gunakan file CSV.");
                    }
                }

                if(count($rows) === 0){
                    json(['success'=>false,'message'=>'File tidak berisi data atau header tidak sesuai. Pastikan header: Nama Tim, Nama Pegawai, Nama Kegiatan, Nama Sub Kegiatan, Target, Satuan, Bulan, Tahun, Deadline.']);
                }

                // Mulai transaksi untuk konsistensi
                mysqli_begin_transaction($conn);

                // Siapkan prepared statement insert (sama seperti saat menyimpan satu record)
                $qIns = mysqli_prepare($conn, "
                    INSERT INTO activity_master
                    (team_id,employee_id,activity_name,sub_activity_name,target_volume,unit,month,year,deadline)
                    VALUES (?,?,?,?,?,?,?,?,?)
                ");

                // prepared statements untuk cek/insert master_kegiatan (agar master ter-sync)
                $chkKeg = mysqli_prepare($conn, "
                    SELECT id FROM master_kegiatan
                    WHERE team_id = ? AND LOWER(nama_kegiatan) = LOWER(?) LIMIT 1
                ");
                $insKeg = mysqli_prepare($conn, "
                    INSERT INTO master_kegiatan (team_id, nama_kegiatan) VALUES (?, ?)
                ");

                // prepared statements untuk cek/insert master_sub_kegiatan (baru ditambahkan)
                $chkSub = mysqli_prepare($conn, "
                    SELECT id FROM master_sub_kegiatan
                    WHERE LOWER(nama_sub_kegiatan) = LOWER(?) LIMIT 1
                ");
                $insSub = mysqli_prepare($conn, "
                    INSERT INTO master_sub_kegiatan (nama_sub_kegiatan) VALUES (?)
                ");

                $inserted = 0;
                $rowNum = 0;
                $warnings = []; // kumpulkan peringatan bila insert sub gagal

                foreach($rows as $r)
                  {
                    $rowNum++;
                    // Ambil nilai dari kunci bahasa Indonesia (snake_case)
                    $namaTim      = isset($r['nama_tim']) ? trim($r['nama_tim']) : (isset($r['Nama Tim']) ? trim($r['Nama Tim']) : '');
                    $namaPegawai  = isset($r['nama_pegawai']) ? trim($r['nama_pegawai']) : (isset($r['Nama Pegawai']) ? trim($r['Nama Pegawai']) : '');
                    $namaKegiatan = isset($r['nama_kegiatan']) ? trim($r['nama_kegiatan']) : (isset($r['Nama Kegiatan']) ? trim($r['Nama Kegiatan']) : '');
                    $namaSub      = isset($r['nama_sub_kegiatan']) ? trim($r['nama_sub_kegiatan']) : (isset($r['Nama Sub Kegiatan']) ? trim($r['Nama Sub Kegiatan']) : (isset($r['sub_kegiatan']) ? trim($r['sub_kegiatan']) : ''));
                    $targetVal    = isset($r['target']) ? $r['target'] : (isset($r['Target']) ? $r['Target'] : (isset($r['target_volume']) ? $r['target_volume'] : 0));
                    $satuanVal    = isset($r['satuan']) ? $r['satuan'] : (isset($r['Satuan']) ? $r['Satuan'] : (isset($r['unit']) ? $r['unit'] : null));
                    $bulanValRaw  = isset($r['bulan']) ? $r['bulan'] : (isset($r['Bulan']) ? $r['Bulan'] : (isset($r['month']) ? $r['month'] : ''));
                    $tahunVal     = isset($r['tahun']) ? $r['tahun'] : (isset($r['Tahun']) ? $r['Tahun'] : (isset($r['year']) ? $r['year'] : null));
                    $deadlineRaw  = isset($r['deadline']) ? $r['deadline'] : (isset($r['Deadline']) ? $r['Deadline'] : (isset($r['tgl']) ? $r['tgl'] : (isset($r['tanggal']) ? $r['tanggal'] : '')));

                    // Skip jika kolom wajib kosong
                    if($namaTim === '' || $namaPegawai === '' || $namaKegiatan === '') {
                        continue;
                    }

                    // proses angka target
                    $target = 0;
                    if($targetVal !== null && $targetVal !== '') $target = (int)preg_replace('/[^0-9\-]/','', (string)$targetVal);

                    // proses bulan (teks atau angka)
                    $monthVal = null;
                    if($bulanValRaw !== null && $bulanValRaw !== ''){
                        if(is_numeric($bulanValRaw)) $monthVal = (int)$bulanValRaw;
                        else {
                            $mapBulan = [
                                'januari'=>1,'februari'=>2,'maret'=>3,'april'=>4,'mei'=>5,'juni'=>6,
                                'juli'=>7,'agustus'=>8,'september'=>9,'oktober'=>10,'november'=>11,'desember'=>12,
                                'january'=>1,'february'=>2,'march'=>3,'april'=>4,'may'=>5,'june'=>6,
                                'july'=>7,'august'=>8,'september'=>9,'october'=>10,'november'=>11,'december'=>12
                            ];
                            $mm = normalizeHeader($bulanValRaw);
                            foreach($mapBulan as $k=>$v){
                                if(strpos($mm, $k) !== false){ $monthVal = $v; break; }
                            }
                        }
                    }

                    // proses deadline (gunakan parseDeadline agar toleran)
                    $deadlineVal = null;
                    if(!empty($deadlineRaw)){
                        $deadlineVal = parseDeadline($deadlineRaw, $hasPhpSpreadsheet);
                    }

                    // Masukkan ke DB (buat tim/pegawai bila perlu)
                    $teamId = getTeamId($conn, $namaTim);
                    $empId  = getEmployeeId($conn, $namaPegawai, $teamId);

                    // Pastikan master_kegiatan ada untuk team ini (cek lalu insert jika perlu)
                    mysqli_stmt_bind_param($chkKeg, "is", $teamId, $namaKegiatan);
                    mysqli_stmt_execute($chkKeg);
                    $resChk = mysqli_stmt_get_result($chkKeg);
                    $existsKeg = false;
                    if($resChk && ($d = mysqli_fetch_assoc($resChk))){
                        $existsKeg = true;
                    }
                    if($resChk) mysqli_free_result($resChk);

                    if(!$existsKeg){
                        mysqli_stmt_bind_param($insKeg, "is", $teamId, $namaKegiatan);
                        mysqli_stmt_execute($insKeg);
                        // ignore errors here; if fails, transaction will rollback later
                    }

                    // ---- Sinkronisasi master_sub_kegiatan (baru ditambahkan) ----
                    if($namaSub !== ''){
                        // cek apakah sudah ada (case-insensitive)
                        mysqli_stmt_bind_param($chkSub, "s", $namaSub);
                        mysqli_stmt_execute($chkSub);
                        $resSub = mysqli_stmt_get_result($chkSub);
                        $existsSub = false;
                        if($resSub && ($ds = mysqli_fetch_assoc($resSub))) $existsSub = true;
                        if($resSub) mysqli_free_result($resSub);

                        if(!$existsSub){
                            // coba insert; jika gagal karena struktur DB (mis. FK atau kolom missing), jangan hentikan proses
                            mysqli_stmt_bind_param($insSub, "s", $namaSub);
                            $okSub = @mysqli_stmt_execute($insSub);
                            if(!$okSub){
                                $errno = mysqli_errno($conn);
                                $err = mysqli_error($conn);
                                $warnings[] = "Baris {$rowNum}: gagal menambahkan master_sub_kegiatan '{$namaSub}' (errno {$errno}) - {$err}";
                                // tidak menghentikan loop; lanjutkan
                            }
                        }
                    }

                    // Bind & execute insert activity_master
                    mysqli_stmt_bind_param($qIns, "iissisiss",
                        $teamId, $empId,
                        $namaKegiatan,
                        $namaSub,
                        $target,
                        $satuanVal,
                        $monthVal,
                        $tahunVal,
                        $deadlineVal
                    );
                    mysqli_stmt_execute($qIns);
                    if(mysqli_errno($conn) === 0) $inserted++;
                }

                // Tutup prepared statement tambahan
                mysqli_stmt_close($qIns);
                mysqli_stmt_close($chkKeg);
                mysqli_stmt_close($insKeg);
                mysqli_stmt_close($chkSub);
                mysqli_stmt_close($insSub);

                // Commit transaksi
                mysqli_commit($conn);

                $resp = ['success'=>true, 'inserted'=>$inserted, 'message'=>"Selesai memproses file. {$inserted} baris berhasil disimpan."];
                if(!empty($warnings)) $resp['warnings'] = $warnings;
                json($resp);
            } catch(Exception $e){
                // rollback apabila terjadi error
                mysqli_rollback($conn);
                json(['success'=>false, 'message'=>'Error memproses file: '.$e->getMessage()]);
            }
        } // end if ada file upload

        // =========================================
        // Jika tidak ada file -> proses 1 baris dari form
        // =========================================
        $teamId = getTeamId($conn, $_POST['team_name']);
        $empId  = getEmployeeId($conn, $_POST['employee_name'], $teamId);
        $target = (int)$_POST['target_volume'];

        // Optional: pastikan master_kegiatan ada (sinkron)
        $namaKegiatanForm = trim($_POST['activity_name']);
        if($namaKegiatanForm !== ''){
            $chk = mysqli_prepare($conn, "SELECT id FROM master_kegiatan WHERE team_id=? AND LOWER(nama_kegiatan)=LOWER(?) LIMIT 1");
            mysqli_stmt_bind_param($chk, "is", $teamId, $namaKegiatanForm);
            mysqli_stmt_execute($chk);
            mysqli_stmt_store_result($chk);
            $exists = mysqli_stmt_num_rows($chk) > 0;
            mysqli_stmt_close($chk);
            if(!$exists){
                $ins = mysqli_prepare($conn, "INSERT INTO master_kegiatan(team_id,nama_kegiatan) VALUES(?,?)");
                mysqli_stmt_bind_param($ins, "is", $teamId, $namaKegiatanForm);
                mysqli_stmt_execute($ins);
                mysqli_stmt_close($ins);
            }
        }

        // Optional: sinkronisasi master_sub_kegiatan dari form
        $namaSubForm = trim($_POST['sub_activity_name'] ?? '');
        if($namaSubForm !== ''){
            $chk = mysqli_prepare($conn, "SELECT id FROM master_sub_kegiatan WHERE LOWER(nama_sub_kegiatan)=LOWER(?) LIMIT 1");
            mysqli_stmt_bind_param($chk, "s", $namaSubForm);
            mysqli_stmt_execute($chk);
            mysqli_stmt_store_result($chk);
            $exists = mysqli_stmt_num_rows($chk) > 0;
            mysqli_stmt_close($chk);
            if(!$exists){
                $ins = mysqli_prepare($conn, "INSERT INTO master_sub_kegiatan(nama_sub_kegiatan) VALUES(?)");
                mysqli_stmt_bind_param($ins, "s", $namaSubForm);
                $ok = @mysqli_stmt_execute($ins);
                if(!$ok){
                    // ignore error but could log; do not stop user flow
                }
                mysqli_stmt_close($ins);
            }
        }

        // parse deadline dari form (toleran)
        $deadlineVal = null;
        if(isset($_POST['deadline']) && trim($_POST['deadline']) !== ''){
            $deadlineVal = parseDeadline($_POST['deadline'], $hasPhpSpreadsheet);
        }

        $q = mysqli_prepare($conn, "
        INSERT INTO activity_master
        (team_id,employee_id,activity_name,sub_activity_name,target_volume,unit,month,year,deadline)
        VALUES (?,?,?,?,?,?,?,?,?)");
        mysqli_stmt_bind_param($q, "iissisiss",
            $teamId, $empId,
            $_POST['activity_name'],
            $_POST['sub_activity_name'],
            $target,
            $_POST['unit'],
            $_POST['month'],
            $_POST['year'],
            $deadlineVal
        );
        mysqli_stmt_execute($q);
        mysqli_stmt_close($q);
        json(['success'=>true, 'message'=>'Data berhasil disimpan.']);
    }

    if($_GET['action']=='update'){
        $teamId = getTeamId($conn, $_POST['team_name']);
        $empId  = getEmployeeId($conn, $_POST['employee_name'], $teamId);
        $target = (int)$_POST['target_volume'];

        // optional: sync master_kegiatan for updated name
        $namaKegiatanForm = trim($_POST['activity_name']);
        if($namaKegiatanForm !== ''){
            $chk = mysqli_prepare($conn, "SELECT id FROM master_kegiatan WHERE team_id=? AND LOWER(nama_kegiatan)=LOWER(?) LIMIT 1");
            mysqli_stmt_bind_param($chk, "is", $teamId, $namaKegiatanForm);
            mysqli_stmt_execute($chk);
            mysqli_stmt_store_result($chk);
            $exists = mysqli_stmt_num_rows($chk) > 0;
            mysqli_stmt_close($chk);
            if(!$exists){
                $ins = mysqli_prepare($conn, "INSERT INTO master_kegiatan(team_id,nama_kegiatan) VALUES(?,?)");
                mysqli_stmt_bind_param($ins, "is", $teamId, $namaKegiatanForm);
                mysqli_stmt_execute($ins);
                mysqli_stmt_close($ins);
            }
        }

        // optional: sync master_sub_kegiatan for updated sub
        $namaSubForm = trim($_POST['sub_activity_name'] ?? '');
        if($namaSubForm !== ''){
            $chk = mysqli_prepare($conn, "SELECT id FROM master_sub_kegiatan WHERE LOWER(nama_sub_kegiatan)=LOWER(?) LIMIT 1");
            mysqli_stmt_bind_param($chk, "s", $namaSubForm);
            mysqli_stmt_execute($chk);
            mysqli_stmt_store_result($chk);
            $exists = mysqli_stmt_num_rows($chk) > 0;
            mysqli_stmt_close($chk);
            if(!$exists){
                $ins = mysqli_prepare($conn, "INSERT INTO master_sub_kegiatan(nama_sub_kegiatan) VALUES(?)");
                mysqli_stmt_bind_param($ins, "s", $namaSubForm);
                mysqli_stmt_execute($ins);
                mysqli_stmt_close($ins);
            }
        }

        // parse deadline dari form (toleran)
        $deadlineVal = null;
        if(isset($_POST['deadline']) && trim($_POST['deadline']) !== ''){
            $deadlineVal = parseDeadline($_POST['deadline'], $hasPhpSpreadsheet);
        }

        $q = mysqli_prepare($conn, "
        UPDATE activity_master SET
        team_id=?, employee_id=?, activity_name=?, sub_activity_name=?,
        target_volume=?, unit=?, month=?, year=?, deadline=?
        WHERE id=?");
        mysqli_stmt_bind_param($q, "iissisissi",
            $teamId, $empId,
            $_POST['activity_name'],
            $_POST['sub_activity_name'],
            $target,
            $_POST['unit'],
            $_POST['month'],
            $_POST['year'],
            $deadlineVal,
            $_POST['id']
        );
        mysqli_stmt_execute($q);
        mysqli_stmt_close($q);
        json(['success'=>true, 'message'=>'Data berhasil diperbarui.']);
    }

    if($_GET['action']=='detail'){
        $q = mysqli_prepare($conn, "
        SELECT am.*, t.name team, e.name employee
        FROM activity_master am
        JOIN teams t ON t.id=am.team_id
        JOIN employees e ON e.id=am.employee_id
        WHERE am.id=?");
        mysqli_stmt_bind_param($q, "i", $_GET['id']);
        mysqli_stmt_execute($q);
        $res = mysqli_stmt_get_result($q);
        $row = mysqli_fetch_assoc($res);
        mysqli_stmt_close($q);
        json($row);
    }

    if($_GET['action']=='list'){
        $r = mysqli_query($conn, "
        SELECT am.id, t.name team, e.name employee,
               am.activity_name, am.sub_activity_name,
               am.target_volume, am.unit, am.month, am.year, am.deadline
        FROM activity_master am
        JOIN teams t ON t.id=am.team_id
        JOIN employees e ON e.id=am.employee_id
        ORDER BY am.id DESC");
        $d = [];
        while($x = mysqli_fetch_assoc($r)) $d[] = $x;
        json($d);
    }

    if($_GET['action']=='delete'){
        $q = mysqli_prepare($conn, "DELETE FROM activity_master WHERE id=?");
        mysqli_stmt_bind_param($q, "i", $_POST['id']);
        mysqli_stmt_execute($q);
        mysqli_stmt_close($q);
        json(['success'=>true, 'message'=>'Data berhasil dihapus.']);
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Admin Tim — Tambah Kegiatan</title>

<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" rel="stylesheet">
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<style>
/* GENERAL */
body, .form-control, .table { font-size: 13px; }
.container{
  width:100%;
  max-width: min(3840px, 98vw);
  padding-left:12px;
  padding-right:12px;
}
/* TABLE */
.table th, .table td { padding: .45rem .6rem; vertical-align: middle; }
.table thead th { font-weight: 600; background: #0d6efd; color: #fff; border-color: transparent; }
.table-responsive { overflow-x:auto; }

/* BUTTONS */
.btn, .btn-sm { font-size: 12px; padding: 6px 10px; }
.btn-action{ padding:4px 7px; font-size:12px; line-height:1; border-radius:4px; }
.btn-action i{ font-size:10px; }

/* FORM */
.form-text small { font-size: 11px; color:#666; }

/* search-like input appearance */
.select-search { height:32px; padding:4px 8px; font-size:13px; }
.select-search::placeholder{ color:#999; }

/* SMALLER SCREENS: stack inputs */
@media (max-width: 767.98px) {
  .form-row-mobile > * { margin-bottom: .5rem; }
  .table thead th { font-size: 12px; }
  .btn, .btn-sm { font-size: 12px; padding: 8px 10px; }
}

/* SweetAlert animation tweak */
@keyframes swal-pop { 0% { transform: scale(0.5); opacity:0 } 60% { transform: scale(1.12); opacity:1 } 100% { transform:scale(1); } }
.swal-success .swal2-success { animation: swal-pop 420ms cubic-bezier(.2,.9,.2,1); }

/* ----------------- Pagination style (match Rekap Kegiatan) ----------------- */
.pagination{
  display:flex;
  gap:6px;
  padding-left:0;
  justify-content:center;
  margin-top:10px;
  margin-bottom:0;
}
.pagination li{ list-style:none; }
.pagination li button{
  border: none;
  background: #ffffff;
  border-radius: 8px;
  padding: 6px 10px;
  font-size: 13px;
  box-shadow: 0 1px 3px rgba(0,0,0,0.08);
  cursor: pointer;
  line-height:1;
  color: #212529;
}
.pagination li button:hover{ background:#f3f6fb; }
.pagination li button.active{
  background: #0d6efd;
  color: #fff;
  box-shadow: 0 2px 6px rgba(13,110,253,0.15);
}
.pagination li button:disabled{
  opacity:0.45;
  cursor: not-allowed;
  background: #fff;
  box-shadow: none;
}
.pagination li button i{ font-size: 14px; vertical-align:middle; }
</style>
</head>
<body class="bg-light">

<?php
// ambil data master untuk dropdown (teams)
$teams_q = mysqli_query($conn, "SELECT name FROM teams ORDER BY name");

// bulan/tahun sekarang untuk filter/placeholder
$bulan = ["","Januari","Februari","Maret","April","Mei","Juni","Juli","Agustus","September","Oktober","November","Desember"];
$curYear = intval(date('Y'));
$curMonth = intval(date('m'));
?>

<div class="container mt-3 mb-5">
  <h4><i class="bi bi-people-fill"></i> Tambah Aktivitas Kegiatan</h4>

  <!-- FORM INPUT -->
  <form id="addForm" class="row g-2 form-row-mobile mt-3" enctype="multipart/form-data">

    <div class="col-12 col-md-4">
      <select name="team_name" id="teamSelect" class="form-control" required aria-label="Pilih Tim">
        <option value="">Pilih Tim</option>
        <?php
        mysqli_data_seek($teams_q,0);
        while($t = mysqli_fetch_assoc($teams_q)){
          $n = htmlspecialchars($t['name']);
          echo "<option value=\"{$n}\">{$n}</option>";
        }
        ?>
      </select>
    </div>

    <div class="col-12 col-md-4">
      <!-- SINGLE INPUT + DATALIST: Pegawai (bisa klik dropdown atau ketik bebas) -->
      <input list="employeeList" id="employeeInput" name="employee_name" class="form-control select-search" placeholder="Cari atau ketik nama pegawai..." required aria-label="Pilih / Ketik Pegawai">
      <datalist id="employeeList">
        <!-- diisi oleh JS via AJAX -->
      </datalist>
    </div>

    <div class="col-12 col-md-4">
      <!-- SINGLE INPUT + DATALIST: Kegiatan (bergantung pada Tim) -->
      <input list="activityList" id="activityInput" name="activity_name" class="form-control select-search" placeholder="Pilih atau ketik kegiatan..." required aria-label="Pilih / Ketik Kegiatan">
      <datalist id="activityList">
        <!-- diisi oleh JS via AJAX saat team dipilih -->
      </datalist>
    </div>

    <div class="col-12 col-md-4">
      <select name="sub_activity_name" id="subSelect" class="form-control" aria-label="Pilih Sub Kegiatan">
        <option value="">Pilih Sub Kegiatan</option>
      </select>
    </div>

    <div class="col-6 col-md-2">
      <input type="number" name="target_volume" class="form-control" min="1" step="1" placeholder="Target" required>
    </div>

    <div class="col-6 col-md-2">
      <input name="unit" class="form-control" placeholder="Satuan (publikasi, dll)">
    </div>

    <div class="col-6 col-md-2">
      <select name="month" class="form-control" required>
        <option value="">Bulan</option>
        <?php
        for ($i = 1; $i <= 12; $i++) {
            $sel = ($i === $curMonth) ? 'selected' : '';
            echo "<option value='$i' $sel>{$bulan[$i]}</option>";
        }
        ?>
      </select>
    </div>

    <div class="col-6 col-md-2">
      <input type="number" name="year" class="form-control" min="2020" max="2100" placeholder="Tahun" value="<?php echo $curYear; ?>" required>
    </div>

    <div class="col-12 col-md-4 form-floating">
      <input type="date" name="deadline" class="form-control" id="deadline">
      <label for="deadline">Deadline Kegiatan</label>
    </div>

    <div class="col-12 col-md-8 d-flex gap-2 align-items-center">
      <!-- Download template: ubah menjadi tombol hijau -->
      <a href="admintim.php?action=download_template" class="btn btn-success btn-sm" title="Download Template Excel">
        <i class="bi bi-download"></i>
      </a>

      <div class="w-100">
        <input type="file" id="excelInput" name="excel_file" class="form-control form-control-sm" accept=".xlsx,.xls,.csv">
        <small class="form-text">Unggah Excel atau CSV untuk input banyak data.</small>
      </div>
    </div>

    <div class="col-12 d-flex justify-content-start mt-2">
      <button type="submit" class="btn btn-success">
        <i class="bi bi-save me-1"></i> Simpan
      </button>
    </div>
  </form>

  <hr>

  <!-- FILTER -->
  <div class="row g-2 mb-2 align-items-center">
    <div class="col-12 col-md-3 position-relative">
      <div class="position-relative">
        <i class="bi bi-search" style="position:absolute; left:10px; top:50%; transform:translateY(-50%); color:#999;"></i>
        <input type="text" id="searchTeam" class="form-control ps-4" placeholder="Cari Nama Tim" aria-label="Cari Nama Tim">
      </div>
    </div>

    <div class="col-6 col-md-2 position-relative">
      <i class="bi bi-calendar3" style="position:absolute; left:10px; top:50%; transform:translateY(-50%); color:#999;"></i>
      <select id="filterYear" class="form-control ps-4" aria-label="Filter Tahun">
        <option value="">Semua Tahun</option>
        <?php
        for ($y = $curYear; $y <= $curYear + 2; $y++) {
            $selected = ($y == $curYear) ? 'selected' : '';
            echo "<option value='$y' $selected>$y</option>";
        }
        ?>
      </select>
    </div>

    <div class="col-6 col-md-2 position-relative">
      <i class="bi bi-calendar" style="position:absolute; left:10px; top:50%; transform:translateY(-50%); color:#999;"></i>
      <select id="filterMonth" class="form-control ps-4" aria-label="Filter Bulan">
        <option value="">Semua Bulan</option>
        <?php
          for($i=1;$i<=12;$i++){
              $selected = ($i === $curMonth) ? 'selected' : '';
              echo "<option value='$i' $selected>{$bulan[$i]}</option>";
          }
        ?>
      </select>
    </div>

    <div class="col-12 col-md-5 text-end">
      <!-- Tombol Hapus Semua dihilangkan sesuai permintaan -->
    </div>
  </div>

  <!-- TABEL -->
  <div class="table-responsive">
    <table class="table table-bordered table-striped table-sm mb-0">
      <thead class="text-center">
        <tr>
          <th>No</th><th>Tim</th><th>Pegawai</th><th>Kegiatan</th><th>Sub</th>
          <th>Target</th><th>Satuan</th><th>Bulan</th><th>Tahun</th><th>Deadline</th><th>Aksi</th>
        </tr>
      </thead>
      <tbody id="data"></tbody>
    </table>
  </div>

  <nav><ul class="pagination" id="pagination"></ul></nav>
</div>

<!-- MODAL EDIT -->
<div class="modal fade" id="editModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg">
    <div class="modal-content">
      <form id="editForm">
        <div class="modal-header bg-warning">
          <h5 class="modal-title"><i class="bi bi-pencil-square"></i> Edit Data</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Tutup"></button>
        </div>
        <div class="modal-body row g-2">
          <input type="hidden" name="id" id="eid">

          <div class="col-12 col-md-4">
            <select id="eteam" name="team_name" class="form-control">
              <option value="">Pilih Tim</option>
              <?php
              mysqli_data_seek($teams_q,0);
              while($t = mysqli_fetch_assoc($teams_q)){
                $n = htmlspecialchars($t['name']);
                echo "<option value=\"{$n}\">{$n}</option>";
              }
              ?>
            </select>
          </div>

          <div class="col-12 col-md-4">
            <!-- Modal: Pegawai input + datalist -->
            <input list="eemployeeList" id="eemployeeInput" name="employee_name" class="form-control select-search" placeholder="Cari atau ketik pegawai..." aria-label="Cari Pegawai (modal)">
            <datalist id="eemployeeList"></datalist>
          </div>

          <div class="col-12 col-md-4">
            <!-- Modal: Kegiatan input + datalist -->
            <input list="eactivityList" id="eactivityInput" name="activity_name" class="form-control select-search" placeholder="Cari atau ketik kegiatan..." aria-label="Cari Kegiatan (modal)">
            <datalist id="eactivityList"></datalist>
          </div>

          <div class="col-12 col-md-4">
            <select id="esub" name="sub_activity_name" class="form-control">
              <option value="">Pilih Sub Kegiatan</option>
            </select>
          </div>

          <div class="col-6 col-md-2"><input type="number" id="etar" name="target_volume" min="1" step="1" class="form-control"></div>
          <div class="col-6 col-md-2"><input id="eunit" name="unit" class="form-control"></div>
          <div class="col-6 col-md-2">
            <select id="emon" name="month" class="form-control">
              <?php for($i=1;$i<=12;$i++) echo "<option value='$i'>{$bulan[$i]}</option>"; ?>
            </select>
          </div>
          <div class="col-6 col-md-2"><input type="number" id="eyear" name="year" class="form-control"></div>
          <div class="col-12 col-md-4"><input type="date" id="edead" name="deadline" class="form-control"></div>
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-secondary btn-sm" data-bs-dismiss="modal"><i class="bi bi-x-circle"></i> Batal</button>
          <button class="btn btn-primary btn-sm" type="submit"><i class="bi bi-arrow-repeat"></i> Update</button>
        </div>
      </form>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<script>
/* Variabel & helper */
const bulanNama=["","Januari","Februari","Maret","April","Mei","Juni","Juli","Agustus","September","Oktober","November","Desember"];
const editModalEl = document.getElementById('editModal');
const modal = new bootstrap.Modal(editModalEl);

let currentPage = 1;
const rowsPerPage = 8;
let tableData = [];
let filteredData = [];

function formatDateIsoToLong(isoDate){
  if(!isoDate || isoDate === "0000-00-00" || isoDate === null) return '-';

  // Jika format YYYY-MM (tanpa hari)
  if(/^\d{4}-\d{2}$/.test(isoDate)){
    const parts = isoDate.split('-');
    const y = parseInt(parts[0],10);
    const m = parseInt(parts[1],10);
    if(!isNaN(m) && !isNaN(y)) return `${bulanNama[m]} ${y}`;
  }

  // Jika disimpan sebagai YYYY-MM-01 karena only month/year -> tampilkan sebagai Bulan Tahun
  if(/^\d{4}-\d{2}-01$/.test(isoDate)){
    const parts = isoDate.split('-');
    const y = parseInt(parts[0],10);
    const m = parseInt(parts[1],10);
    if(!isNaN(m) && !isNaN(y)) return `${bulanNama[m]} ${y}`;
  }

  // Jika format ISO lengkap (YYYY-MM-DD) atau lainnya yang dapat di-parse Date
  const d = new Date(isoDate);
  if(!isNaN(d) && d.toString() !== 'Invalid Date'){
    return `${d.getDate()} ${bulanNama[d.getMonth()+1]} ${d.getFullYear()}`;
  }

  // fallback: tampilkan apa adanya (string yang tidak dikenali)
  return isoDate;
}

/* Load data dari server */
async function load(){
 let r=await fetch('admintim.php?action=list');
 tableData = await r.json();
 applyFilters();
}

/* ------------------- FILTER / TABLE ------------------- */
const searchInput = document.getElementById('searchTeam');
const filterMonth = document.getElementById('filterMonth');
const filterYear  = document.getElementById('filterYear');
searchInput.addEventListener('input', applyFilters);
filterMonth.addEventListener('change', applyFilters);
filterYear.addEventListener('change', applyFilters);

function applyFilters(){
 const searchVal = searchInput.value.toLowerCase();
 const monthVal = filterMonth.value;
 const yearVal  = filterYear.value;
 filteredData = tableData.filter(x=>{
   let matchTeam = x.team.toLowerCase().includes(searchVal);
   let matchMonth = monthVal==='' || String(x.month) === String(monthVal);
   let matchYear  = yearVal===''  || String(x.year) === String(yearVal);
   return matchTeam && matchMonth && matchYear;
 });
 currentPage = 1;
 renderTable();
 renderPagination();
}

function renderTable(){
 let start = (currentPage-1)*rowsPerPage;
 let end = start + rowsPerPage;
 let rows = filteredData.slice(start,end);
 let h='';
 let i = start+1;
 rows.forEach(x=>{
  const deadlineDisplay = x.deadline && x.deadline !== "0000-00-00" ? formatDateIsoToLong(x.deadline) : '-';
  h+=`<tr>
  <td class="text-center">${i++}</td>
  <td>${x.team}</td>
  <td>${x.employee}</td>
  <td>${x.activity_name}</td>
  <td>${x.sub_activity_name||'-'}</td>
  <td class="text-end">${x.target_volume}</td>
  <td>${x.unit||'-'}</td>
  <td class="text-center">${bulanNama[x.month]}</td>
  <td class="text-center">${x.year}</td>
  <td class="text-center">${deadlineDisplay}</td>
  <td class="text-center">
   <button class="btn btn-outline-primary btn-action btn-edit" onclick="edit(${x.id})" title="Edit"><i class="fa-solid fa-pen"></i></button>
   <button class="btn btn-outline-danger btn-action btn-delete" onclick="del(${x.id})" title="Hapus"><i class="fa-solid fa-trash"></i></button>
  </td>
  </tr>`;
 });
 document.getElementById('data').innerHTML=h;
}

/* ----------------- NEW renderPagination: match Rekap Kegiatan UI ----------------- */
function renderPagination(){
  const totalPages = Math.ceil(filteredData.length/rowsPerPage) || 1;
  const paginationEl = document.getElementById('pagination');
  if(totalPages <= 1){
    paginationEl.innerHTML = '';
    return;
  }

  let html = '';

  // Prev button
  html += `<li><button ${currentPage===1 ? 'disabled' : ''} onclick="gotoPage(${currentPage-1})"><i class="bi bi-chevron-left"></i></button></li>`;

  // show up to 3 page numbers centered on current page
  let startPage = Math.max(1, currentPage - 1);
  let endPage = Math.min(totalPages, startPage + 2);
  if(endPage - startPage < 2) startPage = Math.max(1, endPage - 2);

  for(let p = startPage; p <= endPage; p++){
    html += `<li><button class="${p===currentPage ? 'active' : ''}" onclick="gotoPage(${p})">${p}</button></li>`;
  }

  // Next button
  html += `<li><button ${currentPage===totalPages ? 'disabled' : ''} onclick="gotoPage(${currentPage+1})"><i class="bi bi-chevron-right"></i></button></li>`;

  paginationEl.innerHTML = html;
}

function gotoPage(p){
 const totalPages = Math.ceil(filteredData.length/rowsPerPage) || 1;
 if(p<1 || p>totalPages) return;
 currentPage = p;
 renderTable();
 renderPagination();
}

/* ------------------- EDIT / MODAL ------------------- */
async function edit(id){
 let r=await fetch('admintim.php?action=detail&id='+id);
 let x=await r.json();
 document.getElementById('eid').value=x.id;
 document.getElementById('eteam').value = x.team;
 // load all employees into datalist (preselect via input value)
 await loadEmployeesInto('#eteam','#eemployeeInput', null, x.employee);
 // load activities for team into datalist and preselect
 await loadActivitiesInto('#eteam','#eactivityInput', x.team, x.activity_name);
 // now load subs (after activity options exist)
 await loadSubsInto('#eactivityInput','#esub', x.activity_name, x.sub_activity_name);
 document.getElementById('etar').value = x.target_volume;
 document.getElementById('eunit').value = x.unit;
 document.getElementById('emon').value = x.month;
 document.getElementById('eyear').value = x.year;
 // isi deadline ke input date (jika format Y-m-d atau YYYY-MM-01)
 document.getElementById('edead').value = x.deadline && x.deadline !== "0000-00-00" ? x.deadline : '';
 modal.show();
}

/* -------- UX: jika file dipilih -> nonaktifkan input manual -------- */
const excelInput = document.getElementById('excelInput');
const addForm = document.getElementById('addForm');

excelInput.addEventListener('change', function(){
  const hasFile = this.files && this.files.length > 0;
  addForm.querySelectorAll('input, select, textarea').forEach(el=>{
    if(el === excelInput) return;
    if(el.tagName.toLowerCase() === 'button') return;
    el.disabled = hasFile;
    if(hasFile){
      el.removeAttribute('required');
    } else {
      if(el.name === 'team_name' || el.name === 'employee_name' || el.name === 'activity_name' || el.name === 'target_volume' || el.name === 'month' || el.name === 'year'){
        el.setAttribute('required','required');
      }
    }
  });
});

/* ------------------- AJAX helper untuk employees, activities & subs ------------------- */
async function fetchJson(url){
  let r = await fetch(url);
  return await r.json();
}

/*
  Perbaikan utama:
  - cache activities per team (activitiesCache)
  - saat user mengetik di input kegiatan, populate datalist hanya dengan kegiatan tim terpilih yang match input (filter)
  - saat tim berubah, ambil activities dari server dan rebuild datalist
*/
const activitiesCache = {}; // activitiesCache[teamName] = ['Kegiatan A', 'Kegiatan B', ...]

async function loadEmployeesInto(teamSelectSelector, employeeTargetSelector, teamName=null, preselect=null){
  const empTarget = (typeof employeeTargetSelector === 'string') ? document.querySelector(employeeTargetSelector) : employeeTargetSelector;
  if(!empTarget) return;
  try{
    const data = await fetchJson('admintim.php?action=get_employees');
    if(empTarget.tagName.toLowerCase() === 'input' && empTarget.list){
      const listEl = document.getElementById(empTarget.getAttribute('list'));
      if(!listEl) return;
      listEl.innerHTML = ''; // clear
      data.forEach(row=>{
        const opt = document.createElement('option');
        opt.value = row.name;
        listEl.appendChild(opt);
      });
      if(preselect) empTarget.value = preselect;
      else if(!empTarget.value) empTarget.value = '';
    } else if(empTarget.tagName.toLowerCase() === 'select'){
      empTarget.innerHTML = '<option value="">Pilih Pegawai</option>';
      data.forEach(row=>{
        const opt = document.createElement('option');
        opt.value = row.name;
        opt.textContent = row.name;
        empTarget.appendChild(opt);
      });
      if(preselect) empTarget.value = preselect;
    }
  } catch(e){ console.error(e); }
}

/*
  loadActivitiesInto:
   - selalu request server untuk mendapatkan data terbaru untuk team (agar master sinkron)
   - simpan hasil di activitiesCache[team]
   - panggil populateActivityDatalist untuk mengisi datalist (dengan optional filter)
*/
async function loadActivitiesInto(teamSelectSelector, activityTargetSelector, teamName=null, preselect=null){
  const teamEl = (typeof teamSelectSelector === 'string') ? document.querySelector(teamSelectSelector) : teamSelectSelector;
  const actTarget  = (typeof activityTargetSelector === 'string') ? document.querySelector(activityTargetSelector) : activityTargetSelector;
  const team = teamName || (teamEl ? teamEl.value : '');
  if(!actTarget) return;

  // Jika tidak ada team, kosongkan datalist/select
  if(!team){
    if(actTarget.tagName.toLowerCase() === 'input' && actTarget.list){
      const listEl = document.getElementById(actTarget.getAttribute('list'));
      if(listEl) listEl.innerHTML = '';
      actTarget.value = '';
    } else if(actTarget.tagName.toLowerCase() === 'select'){
      actTarget.innerHTML = '<option value="">Pilih Kegiatan</option>';
      actTarget.value = '';
    }
    return;
  }

  try{
    // fetch activities for team
    const data = await fetchJson('admintim.php?action=get_activities&team=' + encodeURIComponent(team));
    // normalize to simple array of strings
    const names = data.map(row => (row.nama_kegiatan || row.name || '').trim()).filter(s => s !== '');
    activitiesCache[team] = names;

    // populate list with all activities (unless caller asked to preselect a value)
    populateActivityDatalist(actTarget, team, '', preselect);
  } catch(e){
    console.error(e);
  }
}

/*
  populateActivityDatalist(actTargetElement, teamName, filterText, preselect)
  - actTargetElement: input element (with .list) or select element
  - filterText: substring to filter (case-insensitive)
*/
function populateActivityDatalist(actTarget, team, filterText='', preselect=null){
  if(!actTarget) return;
  const filterLower = (filterText || '').toLowerCase().trim();
  const items = (activitiesCache[team] || []).filter(n => {
    if(!filterLower) return true;
    return n.toLowerCase().includes(filterLower);
  });

  if(actTarget.tagName.toLowerCase() === 'input' && actTarget.list){
    const listEl = document.getElementById(actTarget.getAttribute('list'));
    if(!listEl) return;
    listEl.innerHTML = '';
    items.forEach(name=>{
      const opt = document.createElement('option');
      opt.value = name;
      listEl.appendChild(opt);
    });
    if(preselect) actTarget.value = preselect;
    else {
      // keep current typed value (don't clear), but if nothing typed and there's exact match, optionally set
      if(!actTarget.value && preselect) actTarget.value = preselect;
    }
  } else if(actTarget.tagName.toLowerCase() === 'select'){
    actTarget.innerHTML = '<option value="">Pilih Kegiatan</option>';
    items.forEach(name=>{
      const opt = document.createElement('option');
      opt.value = name;
      opt.textContent = name;
      actTarget.appendChild(opt);
    });
    if(preselect) actTarget.value = preselect;
  }
}

// load subs for kegiatan into sub select; optional preselect
async function loadSubsInto(activitySelector, subSelectSelector, activityName=null, preselect=null){
  const actEl = (typeof activitySelector === 'string') ? document.querySelector(activitySelector) : activitySelector;
  const subEl = (typeof subSelectSelector === 'string') ? document.querySelector(subSelectSelector) : subSelectSelector;
  const kegiatan = activityName || (actEl ? actEl.value : '');
  if(!subEl) return;
  subEl.innerHTML = '<option value="">Pilih Sub Kegiatan</option>';
  try{
    const url = 'admintim.php?action=get_subs' + (kegiatan ? '&kegiatan=' + encodeURIComponent(kegiatan) : '');
    const data = await fetchJson(url);
    data.forEach(row=>{
      const opt = document.createElement('option');
      opt.value = row.nama_sub_kegiatan || row.name || '';
      opt.textContent = row.nama_sub_kegiatan || row.name || '';
      subEl.appendChild(opt);
    });
    if(preselect) subEl.value = preselect;
  } catch(e){ console.error(e); }
}

/* event listeners */

// ketika tim berubah: refresh employee datalist & activity datalist
document.getElementById('teamSelect').addEventListener('change', function(){
  // reload employee list (global) to ensure up-to-date - harmless and keeps parity with original behaviour
  loadEmployeesInto('#teamSelect','#employeeInput', null, null);

  // reload activities for selected team from server and populate datalist
  loadActivitiesInto('#teamSelect','#activityInput', this.value, null);

  // clear sub
  document.getElementById('subSelect').innerHTML = '<option value="">Pilih Sub Kegiatan</option>';
});

// saat user mengetik di field activity: gunakan cache untuk memfilter dan tampilkan hanya aktivitas yang sesuai dengan tim yang dipilih
document.getElementById('activityInput').addEventListener('input', async function(){
  const team = document.getElementById('teamSelect').value;
  const q = this.value || '';
  if(!team){
    // jika tim belum dipilih -> kosongkan datalist agar tidak muncul semua kegiatan
    const listEl = document.getElementById(this.getAttribute('list'));
    if(listEl) listEl.innerHTML = '';
    return;
  }
  // jika cache belum ada, fetch first
  if(!activitiesCache[team]){
    await loadActivitiesInto('#teamSelect','#activityInput', team, null);
    // loadActivitiesInto memanggil populateActivityDatalist
    // tapi kita juga ingin mem-filter berdasarkan q setelah fetch
    populateActivityDatalist(this, team, q, null);
  } else {
    populateActivityDatalist(this, team, q, null);
  }

  // juga update sub list live (bila cocok / exact selection)
  loadSubsInto('#activityInput','#subSelect', this.value, null);
});

// jika user fokus pada activityInput, pastikan datalist di-populate sesuai tim (lihat UX)
document.getElementById('activityInput').addEventListener('focus', function(){
  const team = document.getElementById('teamSelect').value;
  if(!team){
    // kosongkan agar tidak memunculkan global list
    const listEl = document.getElementById(this.getAttribute('list'));
    if(listEl) listEl.innerHTML = '';
    return;
  }
  // gunakan cache jika ada, jika tidak fetch
  if(activitiesCache[team]){
    populateActivityDatalist(this, team, this.value || '', null);
  } else {
    loadActivitiesInto('#teamSelect','#activityInput', team, null);
  }
});

// modal: ketika tim berubah di modal
document.getElementById('eteam').addEventListener('change', function(){
  loadEmployeesInto('#eteam','#eemployeeInput', null, null);
  loadActivitiesInto('#eteam','#eactivityInput', this.value, null);
  document.getElementById('esub').innerHTML = '<option value="">Pilih Sub Kegiatan</option>';
});

// modal: jika user mengetik kegiatan di modal -> filter datalist modal sesuai tim
document.getElementById('eactivityInput').addEventListener('input', async function(){
  const team = document.getElementById('eteam').value;
  const q = this.value || '';
  if(!team){
    const listEl = document.getElementById(this.getAttribute('list'));
    if(listEl) listEl.innerHTML = '';
    return;
  }
  if(!activitiesCache[team]){
    await loadActivitiesInto('#eteam','#eactivityInput', team, null);
    populateActivityDatalist(this, team, q, null);
  } else {
    populateActivityDatalist(this, team, q, null);
  }
  loadSubsInto('#eactivityInput','#esub', this.value, null);
});

// modal: saat fokus
document.getElementById('eactivityInput').addEventListener('focus', function(){
  const team = document.getElementById('eteam').value;
  if(!team){
    const listEl = document.getElementById(this.getAttribute('list'));
    if(listEl) listEl.innerHTML = '';
    return;
  }
  if(activitiesCache[team]) populateActivityDatalist(this, team, this.value || '', null);
  else loadActivitiesInto('#eteam','#eactivityInput', team, null);
});

/* Simpan */
addForm.onsubmit = async e => {
 e.preventDefault();
 let formData = new FormData(addForm);
 let res = await fetch('admintim.php?action=save',{method:'POST',body:formData});
 let data = await res.json();
 if(data.success){
    if(data.inserted !== undefined){
       let txt = `File berhasil diproses. ${data.inserted} baris disimpan.`;
       if(data.warnings && data.warnings.length){
           txt += "\n\nPeringatan:\n" + data.warnings.join("\n");
       }
       Swal.fire({ icon: 'success', title: 'Berhasil', text: txt, showConfirmButton: false, timer: 2200, timerProgressBar: true, customClass: { popup: 'swal-success' } });
    } else {
       Swal.fire({ icon: 'success', title: 'Berhasil', text: data.message || 'Data berhasil disimpan!', showConfirmButton: false, timer: 1500, timerProgressBar: true, customClass: { popup: 'swal-success' } });
    }
    addForm.reset();
    addForm.querySelectorAll('input, select, textarea').forEach(el=>{ if(el.tagName.toLowerCase() !== 'button') el.disabled = false; });
    // kosongkan datalist/value untuk pegawai & kegiatan
    document.getElementById('employeeInput').value = '';
    document.getElementById('employeeList').innerHTML = '';
    document.getElementById('activityInput').value = '';
    document.getElementById('activityList').innerHTML = '';
    document.getElementById('subSelect').innerHTML = '<option value="">Pilih Sub Kegiatan</option>';
    load();
 } else {
    Swal.fire({ icon: 'error', title: 'Gagal', text: data.message || 'Terjadi kesalahan saat menyimpan.', showConfirmButton: true });
 }
};

/* Update */
const editForm = document.getElementById('editForm');
editForm.onsubmit = async e => {
 e.preventDefault();
 let res = await fetch('admintim.php?action=update',{method:'POST',body:new FormData(editForm)});
 let data = await res.json();
 if(data.success){
    modal.hide();
    await load();
    Swal.fire({ icon: 'success', title: 'Berhasil', text: data.message || 'Data berhasil diperbarui!', showConfirmButton: false, timer: 1500, timerProgressBar: true, customClass: { popup: 'swal-success' } });
 }
};

/* Delete */
async function del(id){
  Swal.fire({ title: 'Yakin ingin menghapus?', text: 'Data tim akan dihapus secara permanen.', icon: 'warning', showCancelButton: true, confirmButtonText: 'Ya, hapus', cancelButtonText: 'Batal', confirmButtonColor: '#d33', reverseButtons: false, focusCancel: true }).then(async (result) => {
    if(result.isConfirmed){
      let fd=new FormData(); fd.append('id',id);
      let res=await fetch('admintim.php?action=delete',{method:'POST',body:fd});
      let data=await res.json();
      if(data.success){
        await load();
        Swal.fire({ icon: 'success', title: 'Terhapus', text: data.message || 'Data berhasil dihapus!', showConfirmButton: false, timer: 1500, timerProgressBar: true, customClass: { popup: 'swal-success' } });
      } else {
        Swal.fire({ icon: 'error', title: 'Gagal', text: data.message || 'Gagal menghapus.' });
      }
    }
  });
}

/* Load awal */
document.addEventListener('DOMContentLoaded', async () => {
  await load();
  // isi datalist pegawai awal (global)
  loadEmployeesInto('#teamSelect','#employeeInput');
});
</script>
</body>
</html>
